// Coretax Document Bulk Downloader - Content Script
(function() {
    'use strict';

    let isExtensionActive = false;
    let selectedDocuments = new Set();
    let downloadQueue = [];
    let isDownloading = false;

    // Wait for the page to load completely
    function waitForTable() {
        const table = document.querySelector('p-table[id="table-el"]');
        if (table) {
            initializeExtension();
        } else {
            setTimeout(waitForTable, 1000);
        }
    }

    function initializeExtension() {
        if (isExtensionActive) return;
        
        console.log('Coretax Bulk Downloader: Initializing...');
        
        // Add extension controls to the page
        addExtensionControls();
        
        // Add checkboxes to each document row
        addCheckboxesToRows();
        
        // Listen for new rows (in case of pagination or dynamic loading)
        observeTableChanges();
        
        isExtensionActive = true;
        console.log('Coretax Bulk Downloader: Initialized successfully');
    }

    function addExtensionControls() {
        // Find the table header area
        const tableHeader = document.querySelector('.p-datatable-header');
        if (!tableHeader) return;

        // Create extension control panel
        const controlPanel = document.createElement('div');
        controlPanel.id = 'coretax-bulk-controls';
        controlPanel.className = 'coretax-bulk-controls';
        
        controlPanel.innerHTML = `
            <div class="coretax-control-group">
                <label class="coretax-checkbox-label">
                    <input type="checkbox" id="select-all-documents" class="coretax-checkbox">
                    <span class="coretax-checkbox-text">Select All</span>
                </label>
                <button id="download-selected" class="coretax-download-btn" disabled>
                    <i class="fa-solid fa-download"></i>
                    Download Selected (<span id="selected-count">0</span>)
                </button>
                <button id="clear-selection" class="coretax-clear-btn">
                    <i class="fa-solid fa-times"></i>
                    Clear Selection
                </button>
            </div>
        `;

        // Insert before the existing controls
        tableHeader.insertBefore(controlPanel, tableHeader.firstChild);

        // Add event listeners
        document.getElementById('select-all-documents').addEventListener('change', handleSelectAll);
        document.getElementById('download-selected').addEventListener('click', handleBulkDownload);
        document.getElementById('clear-selection').addEventListener('click', handleClearSelection);
    }

    function addCheckboxesToRows() {
        const rows = document.querySelectorAll('tbody tr');
        rows.forEach((row, index) => {
            // Skip if checkbox already exists
            if (row.querySelector('.coretax-row-checkbox')) return;

            // Find the first cell (document number cell)
            const firstCell = row.querySelector('td:first-child');
            if (!firstCell) return;

            // Create checkbox container
            const checkboxContainer = document.createElement('div');
            checkboxContainer.className = 'coretax-checkbox-container';
            
            checkboxContainer.innerHTML = `
                <label class="coretax-checkbox-label">
                    <input type="checkbox" class="coretax-row-checkbox" data-row-index="${index}">
                    <span class="coretax-checkbox-custom"></span>
                </label>
            `;

            // Insert checkbox at the beginning of the first cell
            firstCell.insertBefore(checkboxContainer, firstCell.firstChild);

            // Add event listener
            const checkbox = checkboxContainer.querySelector('.coretax-row-checkbox');
            checkbox.addEventListener('change', handleRowSelection);
        });
    }

    function observeTableChanges() {
        const tableBody = document.querySelector('tbody');
        if (!tableBody) return;

        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                if (mutation.type === 'childList') {
                    // Re-add checkboxes to new rows
                    setTimeout(() => {
                        addCheckboxesToRows();
                    }, 100);
                }
            });
        });

        observer.observe(tableBody, {
            childList: true,
            subtree: true
        });
    }

    function handleSelectAll(event) {
        const isChecked = event.target.checked;
        const checkboxes = document.querySelectorAll('.coretax-row-checkbox');
        
        checkboxes.forEach(checkbox => {
            checkbox.checked = isChecked;
            const rowIndex = checkbox.dataset.rowIndex;
            if (isChecked) {
                selectedDocuments.add(rowIndex);
            } else {
                selectedDocuments.delete(rowIndex);
            }
        });

        updateDownloadButton();
    }

    function handleRowSelection(event) {
        const checkbox = event.target;
        const rowIndex = checkbox.dataset.rowIndex;
        
        if (checkbox.checked) {
            selectedDocuments.add(rowIndex);
        } else {
            selectedDocuments.delete(rowIndex);
        }

        // Update select all checkbox
        const selectAllCheckbox = document.getElementById('select-all-documents');
        const totalCheckboxes = document.querySelectorAll('.coretax-row-checkbox').length;
        
        if (selectedDocuments.size === 0) {
            selectAllCheckbox.indeterminate = false;
            selectAllCheckbox.checked = false;
        } else if (selectedDocuments.size === totalCheckboxes) {
            selectAllCheckbox.indeterminate = false;
            selectAllCheckbox.checked = true;
        } else {
            selectAllCheckbox.indeterminate = true;
            selectAllCheckbox.checked = false;
        }

        updateDownloadButton();
    }

    function handleClearSelection() {
        selectedDocuments.clear();
        
        // Uncheck all checkboxes
        document.querySelectorAll('.coretax-row-checkbox').forEach(checkbox => {
            checkbox.checked = false;
        });
        
        const selectAllCheckbox = document.getElementById('select-all-documents');
        selectAllCheckbox.checked = false;
        selectAllCheckbox.indeterminate = false;
        
        updateDownloadButton();
    }

    function updateDownloadButton() {
        const downloadBtn = document.getElementById('download-selected');
        const countSpan = document.getElementById('selected-count');
        
        countSpan.textContent = selectedDocuments.size;
        downloadBtn.disabled = selectedDocuments.size === 0;
    }

    async function handleBulkDownload() {
        if (isDownloading) return;
        
        isDownloading = true;
        const downloadBtn = document.getElementById('download-selected');
        downloadBtn.disabled = true;
        downloadBtn.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i> Downloading...';

        try {
            const rows = document.querySelectorAll('tbody tr');
            const downloadPromises = [];

            for (const rowIndex of selectedDocuments) {
                const row = rows[rowIndex];
                if (!row) continue;

                const downloadBtn = row.querySelector('button[id="ActionDownloadButton"]');
                if (downloadBtn) {
                    // Extract document info for better file naming
                    const docNumber = row.querySelector('td:first-child span').nextSibling.textContent.trim();
                    const docTitle = row.querySelector('td:nth-child(3) span').nextSibling.textContent.trim();
                    
                    downloadPromises.push(downloadDocument(downloadBtn, docNumber, docTitle));
                }
            }

            // Download with delay between each to avoid overwhelming the server
            for (let i = 0; i < downloadPromises.length; i++) {
                await downloadPromises[i];
                if (i < downloadPromises.length - 1) {
                    await new Promise(resolve => setTimeout(resolve, 1000)); // 1 second delay
                }
            }

            // Show success message
            showNotification(`Successfully downloaded ${selectedDocuments.size} documents!`, 'success');
            
            // Clear selection after successful download
            handleClearSelection();

        } catch (error) {
            console.error('Bulk download error:', error);
            showNotification('Error occurred during bulk download', 'error');
        } finally {
            isDownloading = false;
            downloadBtn.disabled = false;
            downloadBtn.innerHTML = '<i class="fa-solid fa-download"></i> Download Selected (<span id="selected-count">0</span>)';
        }
    }

    function downloadDocument(downloadButton, docNumber, docTitle) {
        return new Promise((resolve, reject) => {
            try {
                // Create a temporary click event
                const clickEvent = new MouseEvent('click', {
                    bubbles: true,
                    cancelable: true,
                    view: window
                });

                // Add a listener to detect when download starts
                const originalOnClick = downloadButton.onclick;
                
                downloadButton.onclick = function(event) {
                    if (originalOnClick) {
                        originalOnClick.call(this, event);
                    }
                    
                    // Resolve after a short delay to allow download to start
                    setTimeout(() => {
                        resolve();
                    }, 500);
                };

                // Trigger the click
                downloadButton.dispatchEvent(clickEvent);

                // Restore original onclick
                setTimeout(() => {
                    downloadButton.onclick = originalOnClick;
                }, 1000);

            } catch (error) {
                reject(error);
            }
        });
    }

    function showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `coretax-notification coretax-notification-${type}`;
        notification.textContent = message;

        // Add to page
        document.body.appendChild(notification);

        // Remove after 3 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 3000);
    }

    // Initialize when page loads
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', waitForTable);
    } else {
        waitForTable();
    }

    // Listen for messages from popup
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
        switch (request.action) {
            case 'getStatus':
                sendResponse({
                    selectedCount: selectedDocuments.size,
                    totalCount: document.querySelectorAll('.coretax-row-checkbox').length,
                    isActive: isExtensionActive
                });
                break;
            case 'selectAll':
                const selectAllCheckbox = document.getElementById('select-all-documents');
                if (selectAllCheckbox) {
                    selectAllCheckbox.checked = true;
                    selectAllCheckbox.dispatchEvent(new Event('change'));
                }
                break;
            case 'clearSelection':
                handleClearSelection();
                break;
            case 'downloadSelected':
                handleBulkDownload();
                break;
        }
    });

    // Send status updates to popup
    function sendStatusUpdate() {
        chrome.runtime.sendMessage({
            action: 'statusUpdate',
            data: {
                selectedCount: selectedDocuments.size,
                totalCount: document.querySelectorAll('.coretax-row-checkbox').length,
                isActive: isExtensionActive
            }
        });
    }

    // Update status update calls
    function updateDownloadButton() {
        const downloadBtn = document.getElementById('download-selected');
        const countSpan = document.getElementById('selected-count');
        
        countSpan.textContent = selectedDocuments.size;
        downloadBtn.disabled = selectedDocuments.size === 0;
        
        // Send status update to popup
        sendStatusUpdate();
    }

    // Re-initialize if navigating within SPA
    let lastUrl = location.href;
    new MutationObserver(() => {
        const url = location.href;
        if (url !== lastUrl) {
            lastUrl = url;
            isExtensionActive = false;
            selectedDocuments.clear();
            setTimeout(waitForTable, 1000);
        }
    }).observe(document, { subtree: true, childList: true });

})();
